implementation module CMDatabase

import CMCombinators
import CMTypes
import CMUtilities
import StdiTasks


// Sanity checks
checkUsersDB :: Role (Task Void) -> Task Void
checkUsersDB role task = checkDB readUsersDB 
                                 (any (\user -> user.role == role)) 
                                 task 
                                 ("There are no " +++ printToString role +++ " users added yet")

checkPapersDB :: (Task Void) -> Task Void
checkPapersDB task = checkDB readPapersDB 
                             (not o isEmpty) 
                             task 
                             "There are no papers submitted yet"
                       
checkReviewsDB :: (Task Void) -> Task Void
checkReviewsDB task = checkDB readReviewsDB 
                              (not o isEmpty) 
                              task 
                              "There are no reviews finished yet"
                      
checkDB :: (Task [a]) ([a] -> Bool) (Task Void) String -> Task Void                                                  
checkDB taskDB pred task prompt = taskDB =>> \xs ->
                                  if (pred xs)
                                     task
                                     ([Txt prompt] ?>> ok)	       
                                     

// Storage functions for users
usersId :: DBid [User]
usersId = mkDBid "Users" TxtFile

readUsersDB :: Task [User]
readUsersDB = readDB usersId

readUsersByRoleDB :: Role -> Task [User]
readUsersByRoleDB role = readUsersDB =>>
                         return_V o filter (\x -> x.role == role)
                 
writeUsersDB :: [User] -> Task Void
writeUsersDB users = writeDB usersId users #>>
                     void

lookupUserDB :: UserId -> Task User
lookupUserDB uid_ = readUsersDB =>> 
                    return_V o hd o filter (\x -> x.uid_ == uid_)       

verifyUserDB :: Login -> Task (Maybe User)
verifyUserDB login = readUsersDB =>>
                     return_V o toMaybe o filter (\x -> x.User.login.loginName == login.loginName &&
                                                        x.User.login.password  == login.password)
  where toMaybe [] = Nothing
        toMaybe xs = Just (hd xs)
                               
addUserDB :: User -> Task Void
addUserDB user = readUsersDB =>> \users ->
                 writeUsersDB (users ++ [ {user & uid_ = length users
                                                , new_ = True 
                                          }
                                        ])
                                   
updateUserDB :: UserId (User -> User) -> Task Void
updateUserDB uid upd = readUsersDB =>> \users ->
                       writeUsersDB [  if (user.uid_ == uid)
                                          (upd user)
                                          user
                                    \\ user <- users
                                    ]
                                     
updateUserConflictsDB :: UserId [(PaperId, Mark)] -> Task Void
updateUserConflictsDB uid marks = updateUserDB uid (\user -> {user & marks_ = marks})

initUsersDB :: Task Void
initUsersDB = readUsersDB =>> \users ->
              mapSt (initUserDB users) [("chair", Chair), ("bas", PC), ("thomas", PC)] #>>
              void
              
initUserDB :: [User] (String, Role) -> Task Void              
initUserDB users (name, role) = case any (\user -> user.User.login.loginName == name) users of
                                  False -> addUserDB { uid_   = createDefault
                                                     , new_   = createDefault
                                                     , login  = { loginName = name
                                                                , password  = PasswordBox name
                                                                } 
                                                     , email  = toTextInput (name +++ "@cs.ru.nl")
                                                     , role   = role
                                                     , marks_ = []
                                                     }   
                                  True  -> void
    
    
// Storage functions for papers           
papersId :: DBid [Paper]
papersId = mkDBid "Papers" TxtFile

readPapersDB :: Task [Paper]
readPapersDB = readDB papersId

writePapersDB :: [Paper] -> Task Void
writePapersDB papers = writeDB papersId papers #>>
                       void
                       
clearPapersDB :: Task Void
clearPapersDB = writePapersDB []                       

addPaperDB :: Paper -> Task Void
addPaperDB paper = readPapersDB =>> \papers ->
                   writePapersDB (papers ++ [{paper & pid_ = length papers}])
                    
initPapersDB :: Task Void
initPapersDB = appWorld "initPapersDB" readSubmissions =>> \papers -> 
               clearPapersDB #>>
               writePapersDB papers
  where readSubmissions :: !*World -> (![Paper], !*World)   
        readSubmissions world # (_, file, world) = fopen "abstracts.txt" FReadText world
                              # (string, file)   = freadfile file
                              # (_, world)       = fclose file world 
                              # papers           = toPapers string     
                              = (papers, world)     
          where freadfile :: !*File -> (!String, !*File)
                freadfile file = rec file ""
                  where rec :: !*File String -> (String, !*File)
                        rec file acc # (string, file) = freads file 100
                                     | string == "" = (acc, file)
                                     | otherwise  = rec file (acc +++ string)
                
                toPapers :: String -> [Paper]
                toPapers string # strings = split "\n====\n" string
                                = map toPaper strings
                  where toPaper :: String -> Paper
                        toPaper string # (pid_, string)   = split1 ": " string
                                       # (author, string) = split1 ". " string
                                       # (title, string)  = split1 "    submission   information\nAbstract. " string
                                       # abstract         = string % (0, size string)
                                       = { pid_        = (toInt pid_) - 1
                                         , title       = toTextInput title
                                         , author      = toTextInput author
                                         , affiliation = toTextInput "Anonymous Affiliation"
                                         , email       = toTextInput "iTask@force.nl"
                                         , url         = toTextInput "http://wiki.clean.cs.ru.nl"
                                         , abstract    = toTextArea abstract
                                         }                    
 
 
// Storage functions for reviews
reviewsId :: DBid [Review]
reviewsId = mkDBid "Reviews" TxtFile

readReviewsDB :: Task [Review]
readReviewsDB = readDB reviewsId

writeReviewsDB :: [Review] -> Task Void
writeReviewsDB reviews = writeDB reviewsId reviews #>>
                         void
                         
lookupReviewsDB :: Paper -> Task [Review]
lookupReviewsDB paper = readReviewsDB =>> \reviews ->
                        return_V (filter (\review -> review.Review.paper_ == paper.pid_) reviews)                        

addReviewDB :: PaperId UserId Review -> Task Void
addReviewDB pid_ uid_ review = readReviewsDB =>> \reviews ->
                               writeReviewsDB (reviews ++ [ { review & rid_      = length reviews
                                                                     , paper_    = pid_
                                                                     , reviewer_ = uid_
                                                            }
                                                          ])
                                                 

// Storage functions for judgments
judgmentId :: DBid [Judgment]
judgmentId = mkDBid "Judgment" TxtFile

readJudgmentsDB :: Task [Judgment]
readJudgmentsDB = readDB judgmentId

writeJudgmentsDB :: [Judgment] -> Task Void
writeJudgmentsDB judgments = writeDB judgmentId judgments #>>
                             void
                             
addJudgmentDB :: Judgment -> Task Void
addJudgmentDB judgment = readJudgmentsDB =>> \judgments ->
                         writeJudgmentsDB (judgments ++ [{judgment & jid_ = length judgments}])
                         
                 
// Storage functions for assigned reviewers                         
assignedReviewerId :: DBid [AssignedReviewer]
assignedReviewerId = mkDBid "AssignedReviewer" TxtFile

readAssignedReviewersDB :: Task [AssignedReviewer]                    
readAssignedReviewersDB = readDB assignedReviewerId

writeAssignedReviewersDB :: [AssignedReviewer] -> Task Void
writeAssignedReviewersDB assignedReviewers = writeDB assignedReviewerId assignedReviewers #>>
                                             void    
                                             
addAssignedReviewersDB :: [AssignedReviewer] -> Task Void
addAssignedReviewersDB assignedReviewers = readDB assignedReviewerId =>> \assignedReviewers` ->
                                           writeAssignedReviewersDB (assignedReviewers` ++ assignedReviewers)                                                                                      
                       
splitAssignedReviewersDB :: [Paper] -> Task ([Paper], [Paper], [Paper], [Paper])
splitAssignedReviewersDB papers = readJudgmentsDB =>> \judgments ->
                                  readAssignedReviewersDB =>> \assignedReviewers ->
                                  mapSt (\assignedReviewer -> getWorkflowStatus assignedReviewer.handle) assignedReviewers =>> \statuses ->
                                  return_V (split judgments assignedReviewers [  (assignedReviewer.AssignedReviewer.paper_, status) 
                                                                              \\ assignedReviewer <- assignedReviewers 
                                                                              &   status <- statuses 
                                                                              ])
  where split :: [Judgment] [AssignedReviewer] [(PaperId, WorkflowStatus)] -> ([Paper], [Paper], [Paper], [Paper])
        split judgments assignedReviewers statuses # (notAssigned, assigned) = splitBy (\paper -> isMember paper.pid_ 
                                                                                                           (map (\x -> x.AssignedReviewer.paper_) assignedReviewers)
                                                                                       ) papers
                                                   # (notReviewed, reviewed) = splitBy (\paper -> all (\(_, status) -> status == WflFinished || status == WflDeleted) 
                                                                                                      (filter (\(paper_, _) -> paper_ == paper.pid_) statuses)
                                                                                       ) assigned
                                                   # (notJudged, judged)     = splitBy (\paper -> isMember paper.pid_
                                                                                                           (map (\x -> x.Judgment.paper_) judgments)
                                                                                       ) reviewed
                                                   = (notAssigned, notReviewed, notJudged, judged)